<?php
// Enhanced API endpoint for web search functionality
// This file provides web search, webpage visiting, screenshot taking, and advanced scraping capabilities

require_once 'enhanced-web-search.php';

// Set content type to JSON
header('Content-Type: application/json');
header('X-Content-Type-Options: nosniff');

// Security headers
header('X-Frame-Options: DENY');
header('X-XSS-Protection: 1; mode=block');
header('Strict-Transport-Security: max-age=31536000; includeSubDomains');

// Allow cross-origin requests (adjust as needed for security)
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');
header('Access-Control-Max-Age: 86400');

// Handle preflight requests
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

try {
    // Validate request method
    if ($_SERVER['REQUEST_METHOD'] !== 'GET' && $_SERVER['REQUEST_METHOD'] !== 'POST') {
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed. Use GET or POST.']);
        exit();
    }

    // Determine the action based on the endpoint or parameter
    $action = isset($_GET['action']) ? $_GET['action'] : (isset($_POST['action']) ? $_POST['action'] : 'search');

    // Initialize the WebSearchManager
    $searchManager = new WebSearchManager();

    switch ($action) {
        case 'search':
            // Get search query from request parameters
            $query = '';
            $limit = 5; // Default limit

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $query = isset($_GET['q']) ? trim($_GET['q']) : (isset($_GET['query']) ? trim($_GET['query']) : '');
                $limit = isset($_GET['limit']) ? (int)$_GET['limit'] : 5;
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $query = isset($input['q']) ? trim($input['q']) : (isset($input['query']) ? trim($input['query']) : (isset($_POST['q']) ? trim($_POST['q']) : (isset($_POST['query']) ? trim($_POST['query']) : '')));
                $limit = isset($input['limit']) ? (int)$input['limit'] : (isset($_POST['limit']) ? (int)$_POST['limit'] : 5);
            }

            // Validate input
            if (empty($query)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing search query parameter "q" or "query"']);
                exit();
            }

            // Additional validation and sanitization
            $query = trim($query);

            // Check for potentially dangerous characters or patterns
            if (preg_match('/(<script|javascript:|vbscript:|onload|onerror|<iframe|<object|<embed|<form|<input)/i', $query)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid search query. Contains potentially dangerous content.']);
                exit();
            }

            // Sanitize input - remove HTML tags and encode special characters
            $query = htmlspecialchars(strip_tags($query), ENT_QUOTES, 'UTF-8');

            // Validate query length
            if (strlen($query) > 500) {
                http_response_code(400);
                echo json_encode(['error' => 'Search query is too long. Maximum length is 500 characters.']);
                exit();
            }

            // Validate limit parameter
            if (!is_numeric($limit) || $limit < 1 || $limit > 20) {
                $limit = 5; // Default to 5 if invalid
            }

            // Perform the search
            $results = $searchManager->search($query, $limit);

            // Check if search was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'visit':
            // Get URL to visit
            $url = '';

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
            }

            // Validate input
            if (empty($url)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Visit the page
            $results = $searchManager->visitPage($url);

            // Check if visit was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'screenshot':
            // Get URL for screenshot
            $url = '';

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
            }

            // Validate input
            if (empty($url)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Take screenshot
            $results = $searchManager->takeScreenshot($url);

            // Check if screenshot was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'scrape':
            // Get URL to scrape
            $url = '';
            $selectors = []; // Optional custom selectors

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
                $selectorsParam = isset($_GET['selectors']) ? $_GET['selectors'] : '';
                if (!empty($selectorsParam)) {
                    $selectors = json_decode($selectorsParam, true);
                }
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
                $selectors = isset($input['selectors']) ? $input['selectors'] : (isset($_POST['selectors']) ? json_decode($_POST['selectors'], true) : []);
            }

            // Validate input
            if (empty($url)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Perform scraping
            if (!empty($selectors)) {
                $results = $searchManager->advancedScrape($url, $selectors);
            } else {
                $results = $searchManager->scrapeUrl($url);
            }

            // Check if scrape was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'console_logs':
            // Get URL to retrieve console logs from
            $url = '';

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
            }

            // Validate input
            if (empty($url)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Get console logs
            $results = $searchManager->getConsoleLogs($url);

            // Check if operation was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'execute_js':
            // Get URL and JavaScript to execute
            $url = '';
            $javascript = '';

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
                $javascript = isset($_GET['javascript']) ? $_GET['javascript'] : '';
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
                $javascript = isset($input['javascript']) ? $input['javascript'] : (isset($_POST['javascript']) ? $_POST['javascript'] : '');
            }

            // Validate input
            if (empty($url) || empty($javascript)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL or JavaScript parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Execute JavaScript
            $results = $searchManager->executeJavaScript($url, $javascript);

            // Check if operation was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'interact':
            // Get URL and interactions to perform
            $url = '';
            $interactions = [];

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
                $interactionsParam = isset($_GET['interactions']) ? $_GET['interactions'] : '';
                if (!empty($interactionsParam)) {
                    $interactions = json_decode($interactionsParam, true);
                }
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
                $interactions = isset($input['interactions']) ? $input['interactions'] : (isset($_POST['interactions']) ? json_decode($_POST['interactions'], true) : []);
            }

            // Validate input
            if (empty($url) || empty($interactions)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL or interactions parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Perform interactions
            $results = $searchManager->interactWithPage($url, $interactions);

            // Check if operation was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'performance_metrics':
            // Get URL to analyze
            $url = '';

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
            }

            // Validate input
            if (empty($url)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Get performance metrics
            $results = $searchManager->getPerformanceMetrics($url);

            // Check if operation was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'workflow':
            // Get workflow steps
            $steps = [];

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $stepsParam = isset($_GET['steps']) ? $_GET['steps'] : '';
                if (!empty($stepsParam)) {
                    $steps = json_decode($stepsParam, true);
                }
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $steps = isset($input['steps']) ? $input['steps'] : (isset($_POST['steps']) ? json_decode($_POST['steps'], true) : []);
            }

            // Validate input
            if (empty($steps)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing steps parameter']);
                exit();
            }

            // Execute workflow
            $results = $searchManager->executeWorkflow($steps);

            // Check if operation was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'form_submit':
            // Get URL and form data
            $url = '';
            $formData = [];
            $formSelector = 'form'; // default form selector

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
                $formDataParam = isset($_GET['form_data']) ? $_GET['form_data'] : '';
                if (!empty($formDataParam)) {
                    $formData = json_decode($formDataParam, true);
                }
                $formSelector = isset($_GET['form_selector']) ? $_GET['form_selector'] : 'form';
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
                $formData = isset($input['form_data']) ? $input['form_data'] : (isset($_POST['form_data']) ? json_decode($_POST['form_data'], true) : []);
                $formSelector = isset($input['form_selector']) ? $input['form_selector'] : (isset($_POST['form_selector']) ? $_POST['form_selector'] : 'form');
            }

            // Validate input
            if (empty($url) || empty($formData)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL or form data parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Submit form
            $results = $searchManager->submitForm($url, $formData, $formSelector);

            // Check if operation was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'cookies':
            // Get URL, operation, and cookie data
            $url = '';
            $operation = 'get'; // default operation
            $cookieData = [];

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
                $operation = isset($_GET['operation']) ? $_GET['operation'] : 'get';
                $cookieDataParam = isset($_GET['cookie_data']) ? $_GET['cookie_data'] : '';
                if (!empty($cookieDataParam)) {
                    $cookieData = json_decode($cookieDataParam, true);
                }
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
                $operation = isset($input['operation']) ? $input['operation'] : (isset($_POST['operation']) ? $_POST['operation'] : 'get');
                $cookieData = isset($input['cookie_data']) ? $input['cookie_data'] : (isset($_POST['cookie_data']) ? json_decode($_POST['cookie_data'], true) : []);
            }

            // Validate input
            if (empty($url)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Manage cookies
            $results = $searchManager->manageCookies($url, $operation, $cookieData);

            // Check if operation was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'login':
            // Get URL, username, and password
            $url = '';
            $username = '';
            $password = '';
            $selectors = [];

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
                $username = isset($_GET['username']) ? trim($_GET['username']) : '';
                $password = isset($_GET['password']) ? trim($_GET['password']) : '';
                $selectorsParam = isset($_GET['selectors']) ? $_GET['selectors'] : '';
                if (!empty($selectorsParam)) {
                    $selectors = json_decode($selectorsParam, true);
                }
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
                $username = isset($input['username']) ? trim($input['username']) : (isset($_POST['username']) ? trim($_POST['username']) : '');
                $password = isset($input['password']) ? trim($input['password']) : (isset($_POST['password']) ? trim($_POST['password']) : '');
                $selectors = isset($input['selectors']) ? $input['selectors'] : (isset($_POST['selectors']) ? json_decode($_POST['selectors'], true) : []);
            }

            // Validate input
            if (empty($url) || empty($username) || empty($password)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL, username, or password parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Perform login
            $results = $searchManager->login($url, $username, $password, $selectors);

            // Check if operation was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        case 'wait_for':
            // Get URL and conditions
            $url = '';
            $conditions = [];
            $timeout = 10; // default timeout

            if ($_SERVER['REQUEST_METHOD'] === 'GET') {
                $url = isset($_GET['url']) ? trim($_GET['url']) : '';
                $conditionsParam = isset($_GET['conditions']) ? $_GET['conditions'] : '';
                if (!empty($conditionsParam)) {
                    $conditions = json_decode($conditionsParam, true);
                }
                $timeout = isset($_GET['timeout']) ? (int)$_GET['timeout'] : 10;
            } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
                $input = json_decode(file_get_contents('php://input'), true);
                $url = isset($input['url']) ? trim($input['url']) : (isset($_POST['url']) ? trim($_POST['url']) : '');
                $conditions = isset($input['conditions']) ? $input['conditions'] : (isset($_POST['conditions']) ? json_decode($_POST['conditions'], true) : []);
                $timeout = isset($input['timeout']) ? (int)$input['timeout'] : (isset($_POST['timeout']) ? (int)$_POST['timeout'] : 10);
            }

            // Validate input
            if (empty($url) || empty($conditions)) {
                http_response_code(400);
                echo json_encode(['error' => 'Missing URL or conditions parameter']);
                exit();
            }

            // Validate URL format
            if (!filter_var($url, FILTER_VALIDATE_URL)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid URL format']);
                exit();
            }

            // Wait for conditions
            $results = $searchManager->waitForConditions($url, $conditions, $timeout);

            // Check if operation was successful
            if (isset($results['success']) && $results['success'] === false) {
                http_response_code(400); // Bad Request
            } else {
                http_response_code(200); // OK
            }

            break;

        default:
            http_response_code(400);
            echo json_encode(['error' => 'Invalid action. Use: search, visit, screenshot, scrape, console_logs, execute_js, interact, performance_metrics, workflow, form_submit, cookies, login, or wait_for']);
            exit();
    }

    // Return the results as JSON
    echo json_encode($results, JSON_PRETTY_PRINT);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Internal server error',
        'message' => $e->getMessage()
    ]);
}