<?php

// Require Composer autoload
if (file_exists('../vendor/autoload.php')) {
    require_once '../vendor/autoload.php';
} elseif (file_exists('vendor/autoload.php')) {
    require_once 'vendor/autoload.php';
} else {
    throw new Exception('Composer autoload not found. Please run "composer install" first.');
}

use Goutte\Client as GoutteClient;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Component\DomCrawler\Crawler as DomCrawler;
use Symfony\Component\Panther\Client as PantherClient;
use Facebook\WebDriver\WebDriverBy;

/**
 * Enhanced WebSearchManager Class
 * Handles web searches, content scraping, page visiting, and screenshot taking
 */
class WebSearchManager {
    private $client;
    private $cacheDir;
    private $cacheExpiration; // in seconds

    /**
     * Class constants
     */
    private $searchApiKey;
    private $searchEngineId;

    public function __construct($cacheDir = '../cache/search', $cacheExpiration = 3600) {
        $this->client = new GoutteClient(HttpClient::create([
            'headers' => [
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                'Accept-Language' => 'en-US,en;q=0.9',
            ],
            'timeout' => 30,
        ]));

        $this->cacheDir = rtrim($cacheDir, '/');
        $this->cacheExpiration = $cacheExpiration;

        // Set Google Custom Search API credentials
        $this->searchApiKey = "AIzaSyCPR3fFs89C3KoUqlfQ3D7fuY_qE6QMtq8";
        $this->searchEngineId = "b5c87db0a2743426b";

        // Create cache directory if it doesn't exist
        if (!file_exists($this->cacheDir)) {
            mkdir($this->cacheDir, 0755, true);
        }
    }

    /**
     * Log a message to the search log file
     *
     * @param string $message The message to log
     * @param mixed $data Optional data to include
     */
    private function logSearch($message, $data = null) {
        $logDir = '../logs';
        if (!file_exists($logDir)) {
            mkdir($logDir, 0755, true);
        }

        $logFile = $logDir . '/search_log.txt';
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] $message";

        if ($data !== null) {
            $logMessage .= "\nData: " . json_encode($data, JSON_PRETTY_PRINT);
        }

        $logMessage .= "\n" . str_repeat('-', 80) . "\n";

        // Append to log file
        file_put_contents($logFile, $logMessage, FILE_APPEND);
    }

    /**
     * Search the web for information
     *
     * @param string $query The search query
     * @param int $limit Maximum number of results to return
     * @return array Search results
     */
    public function search($query, $limit = 5) {
        $this->logSearch("SEARCH START: Query='$query', Limit=$limit");

        try {
            // Step 1: Get search results from Google Custom Search API
            $this->logSearch("Step 1: Getting initial search results from Google API");

            $searchResults = $this->getGoogleApiResults($query, $limit);

            $this->logSearch("Step 1 Complete: Initial results obtained", [
                'count' => count($searchResults['results']),
                'sample' => array_slice($searchResults['results'], 0, 2)
            ]);

            // Step 2: Scrape content from top results
            $this->logSearch("Step 2: Scraping content from top results");
            $enhancedResults = [];
            foreach ($searchResults['results'] as $index => $result) {
                if ($index >= 5) break; // Limit to top 5 for scraping

                // Get the URL
                $url = $result['link'];
                $this->logSearch("Step 2.$index: Scraping URL: $url");

                try {
                    // Scrape the page
                    $scrapedContent = $this->scrapeUrl($url);

                    $contentLength = strlen($scrapedContent['content'] ?? '');
                    $this->logSearch("Step 2.$index: Scraping completed for URL: $url", [
                        'content_length' => $contentLength,
                        'success' => $scrapedContent['success'],
                        'title' => $scrapedContent['title']
                    ]);

                    // Save the relevant content
                    $enhancedResults[] = [
                        'title' => $result['title'],
                        'link' => $url,
                        'snippet' => $result['snippet'],
                        'content' => $scrapedContent['content'] ?? 'No content available',
                        'html_content' => $scrapedContent['html_content'] ?? '',
                        'footer_content' => $scrapedContent['footer_content'] ?? ''
                    ];
                } catch (Exception $e) {
                    $this->logSearch("Step 2.$index: Scraping failed for URL: $url", [
                        'error' => $e->getMessage()
                    ]);

                    // Still add the result but without enhanced content
                    $enhancedResults[] = [
                        'title' => $result['title'],
                        'link' => $url,
                        'snippet' => $result['snippet'],
                        'content' => $result['snippet'], // Fall back to snippet
                        'html_content' => '<html><body>' . $result['snippet'] . '</body></html>', // Basic HTML wrapper around snippet
                        'footer_content' => '' // No footer content available
                    ];
                }
            }

            // Step 3: Process and summarize the content
            $this->logSearch("Step 3: Processing and summarizing content");
            $processedResults = $this->processResults($query, $enhancedResults);
            $this->logSearch("Step 3 Complete: Results processed", [
                'processed_count' => count($processedResults),
                'sample' => array_slice($processedResults, 0, 2)
            ]);

            // Step 4: Return the enhanced search results
            $finalResult = [
                'success' => true,
                'query' => $query,
                'results' => $processedResults,
                'timestamp' => date('Y-m-d H:i:s')
            ];

            $this->logSearch("SEARCH COMPLETE: Query='$query'", [
                'status' => 'success',
                'result_count' => count($processedResults)
            ]);

            return $finalResult;
        } catch (Exception $e) {
            $errorMsg = "SEARCH ERROR: " . $e->getMessage();
            $this->logSearch($errorMsg, [
                'query' => $query,
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
                'query' => $query
            ];
        }
    }

    /**
     * Get search results from Google Custom Search API
     *
     * @param string $query The search query
     * @param int $limit Maximum number of results to return
     * @return array Search results from Google API
     */
    private function getGoogleApiResults($query, $limit = 5) {
        $this->logSearch("getGoogleApiResults: Searching Google for '$query'");

        try {
            // Construct the Google Custom Search API URL
            $apiUrl = "https://www.googleapis.com/customsearch/v1?key={$this->searchApiKey}&cx={$this->searchEngineId}&q=" . urlencode($query) . "&num={$limit}";

            $this->logSearch("getGoogleApiResults: Using API URL: $apiUrl");

            // Make the HTTP request to Google API
            $response = file_get_contents($apiUrl);

            if ($response === false) {
                throw new Exception("Failed to get response from Google API");
            }

            // Parse the JSON response
            $data = json_decode($response, true);

            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("Failed to parse JSON response: " . json_last_error_msg());
            }

            $this->logSearch("getGoogleApiResults: API response parsed successfully");

            // Extract search results
            $results = [];

            if (isset($data['items']) && is_array($data['items'])) {
                foreach ($data['items'] as $item) {
                    $results[] = [
                        'title' => $item['title'] ?? 'No title',
                        'link' => $item['link'] ?? '',
                        'snippet' => $item['snippet'] ?? '',
                        'source' => 'google_api'
                    ];
                }
            }

            $this->logSearch("getGoogleApiResults: Extracted " . count($results) . " results");

            // Format and return the results
            return [
                'success' => true,
                'query' => $query,
                'results' => array_slice($results, 0, $limit),
                'source' => 'google_api',
                'timestamp' => date('Y-m-d H:i:s')
            ];

        } catch (Exception $e) {
            $this->logSearch("getGoogleApiResults: Error occurred while searching Google API", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Return error result instead of mock data
            return [
                'success' => false,
                'query' => $query,
                'results' => [],
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Scrape content from a URL (now supports JS-rendered pages, bot/captcha detection, screenshots, recursion)
     *
     * @param string $url The URL to scrape
     * @param int $depth Recursion depth (default 1)
     * @param array $visited Internal: visited URLs
     * @return array The scraped content
     */
    public function scrapeUrl($url, $depth = 1, &$visited = []) {
        $result = [
            'url' => $url,
            'success' => false,
            'title' => '',
            'description' => '',
            'content' => '',
            'footer_content' => '',
            'html_content' => '',
            'timestamp' => date('Y-m-d H:i:s'),
            'screenshot_base64' => '',
            'bot_detection' => [
                'recaptcha' => false,
                'js_challenge' => false,
                'cookie_challenge' => false,
                'other' => false
            ],
            'links' => [],
            'children' => []
        ];
        $visited[$url] = true;
        try {
            // Try Goutte first for speed
            try {
                $crawler = $this->client->request('GET', $url);
                $title = $crawler->filter('title')->text('');
                if (empty($title)) {
                    $title = $crawler->filter('h1')->text('');
                }
                $description = '';
                $crawler->filter('meta[name="description"]')->each(function (DomCrawler $node) use (&$description) {
                    $description = $node->attr('content');
                });
                $mainContent = $this->getCleanText($crawler);
                $footerContent = '';
                $footerSelectors = ['footer', '.footer', '#footer', '.site-footer', '.page-footer', '.main-footer', '.global-footer'];
                foreach ($footerSelectors as $selector) {
                    if ($crawler->filter($selector)->count() > 0) {
                        $footerContent = $crawler->filter($selector)->text();
                        break;
                    }
                }
                $fullHtml = '';
                try {
                    $html = $crawler->html();
                    if (!empty($html)) {
                        $fullHtml = $html;
                    }
                } catch (Exception $e) {}
                $result['success'] = true;
                $result['title'] = $title;
                $result['description'] = $description;
                $result['content'] = trim(preg_replace('/\s+/', ' ', $mainContent));
                $result['footer_content'] = trim(preg_replace('/\s+/', ' ', $footerContent));
                $result['html_content'] = $fullHtml;
                // Extract links
                $result['links'] = $crawler->filter('a')->each(function (DomCrawler $node) use ($url) {
                    $href = $node->attr('href');
                    if ($href) {
                        return $this->makeAbsoluteUrl($url, $href);
                    }
                    return null;
                });
                $result['links'] = array_filter($result['links']);
            } catch (Exception $e) {
                // Goutte failed (maybe JS site), try Panther
                $pantherClient = PantherClient::createChromeClient();
                $browser = $pantherClient->request('GET', $url);
                // Wait for JS rendering (wait for body)
                $pantherClient->waitFor('#app, body, html', 10);
                $title = $browser->filter('title')->text('');
                if (empty($title)) {
                    $title = $browser->filter('h1')->text('');
                }
                $description = '';
                $browser->filter('meta[name="description"]')->each(function (DomCrawler $node) use (&$description) {
                    $description = $node->attr('content');
                });
                $mainContent = $this->getCleanText($browser);
                $footerContent = '';
                $footerSelectors = ['footer', '.footer', '#footer', '.site-footer', '.page-footer', '.main-footer', '.global-footer'];
                foreach ($footerSelectors as $selector) {
                    if ($browser->filter($selector)->count() > 0) {
                        $footerContent = $browser->filter($selector)->text();
                        break;
                    }
                }
                $fullHtml = '';
                try {
                    $html = $browser->html();
                    if (!empty($html)) {
                        $fullHtml = $html;
                    }
                } catch (Exception $e) {}
                // Screenshot
                try {
                    $screenshot = $pantherClient->takeScreenshot();
                    $result['screenshot_base64'] = base64_encode($screenshot);
                } catch (Exception $e) {
                    $result['screenshot_base64'] = '';
                }
                // Bot/captcha detection
                $htmlLower = strtolower($fullHtml);
                if (strpos($htmlLower, 'recaptcha') !== false || strpos($htmlLower, 'g-recaptcha') !== false) {
                    $result['bot_detection']['recaptcha'] = true;
                }
                if (strpos($htmlLower, 'cf-chl-captcha') !== false || strpos($htmlLower, 'js-challenge') !== false) {
                    $result['bot_detection']['js_challenge'] = true;
                }
                if (strpos($htmlLower, 'cookie') !== false && strpos($htmlLower, 'consent') !== false) {
                    $result['bot_detection']['cookie_challenge'] = true;
                }
                $result['success'] = true;
                $result['title'] = $title;
                $result['description'] = $description;
                $result['content'] = trim(preg_replace('/\s+/', ' ', $mainContent));
                $result['footer_content'] = trim(preg_replace('/\s+/', ' ', $footerContent));
                $result['html_content'] = $fullHtml;
                // Extract links
                $result['links'] = $browser->filter('a')->each(function (DomCrawler $node) use ($url) {
                    $href = $node->attr('href');
                    if ($href) {
                        return $this->makeAbsoluteUrl($url, $href);
                    }
                    return null;
                });
                $result['links'] = array_filter($result['links']);
            }
            // Recursion (scrape links)
            if ($depth > 1) {
                $maxChildren = 5; // Limit recursion for performance
                $children = [];
                $count = 0;
                foreach ($result['links'] as $link) {
                    if (!isset($visited[$link]) && $count < $maxChildren) {
                        $children[] = $this->scrapeUrl($link, $depth - 1, $visited);
                        $count++;
                    }
                }
                $result['children'] = $children;
            }
        } catch (Exception $e) {
            $this->logSearch("scrapeUrl: Error scraping URL: $url", [
                'error' => $e->getMessage()
            ]);
            // Return error result instead of mock data
            return [
                'url' => $url,
                'success' => false,
                'title' => 'Error scraping page',
                'description' => '',
                'content' => 'Failed to scrape content from the URL: ' . $e->getMessage(),
                'footer_content' => '',
                'html_content' => '',
                'timestamp' => date('Y-m-d H:i:s'),
                'screenshot_base64' => '',
                'bot_detection' => [
                    'recaptcha' => false,
                    'js_challenge' => false,
                    'cookie_challenge' => false,
                    'other' => false
                ],
                'links' => [],
                'children' => []
            ];
        }
        return $result;
    }

    /**
     * Visit a webpage and return its content without storing in search results
     *
     * @param string $url The URL to visit
     * @return array The page content
     */
    public function visitPage($url) {
        $this->logSearch("visitPage: Visiting URL: $url");
        return $this->scrapeUrl($url);
    }

    /**
     * Take a screenshot of a webpage
     *
     * @param string $url The URL to take screenshot of
     * @return array The screenshot result
     */
    public function takeScreenshot($url) {
        $this->logSearch("takeScreenshot: Taking screenshot of URL: $url");
        
        try {
            $pantherClient = PantherClient::createChromeClient();
            $browser = $pantherClient->request('GET', $url);
            
            // Wait for page to load
            $pantherClient->waitFor('#app, body, html', 10);
            
            // Take screenshot
            $screenshot = $pantherClient->takeScreenshot();
            $screenshotBase64 = base64_encode($screenshot);
            
            $result = [
                'success' => true,
                'url' => $url,
                'screenshot_base64' => $screenshotBase64,
                'timestamp' => date('Y-m-d H:i:s')
            ];
            
            $this->logSearch("takeScreenshot: Successfully took screenshot of URL: $url");
            return $result;
            
        } catch (Exception $e) {
            $this->logSearch("takeScreenshot: Error taking screenshot of URL: $url", [
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'url' => $url,
                'screenshot_base64' => '',
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Perform advanced scraping with custom selectors
     *
     * @param string $url The URL to scrape
     * @param array $selectors Custom selectors to extract specific content
     * @return array The scraped content with custom selections
     */
    public function advancedScrape($url, $selectors = []) {
        $this->logSearch("advancedScrape: Advanced scraping of URL: $url");
        
        try {
            // Start with basic scraping
            $result = $this->scrapeUrl($url);
            
            if ($result['success'] && !empty($selectors)) {
                $pantherClient = PantherClient::createChromeClient();
                $browser = $pantherClient->request('GET', $url);
                
                // Wait for page to load
                $pantherClient->waitFor('#app, body, html', 10);
                
                // Extract custom content based on selectors
                $customContent = [];
                foreach ($selectors as $name => $selector) {
                    try {
                        $elements = $browser->filter($selector);
                        $content = [];
                        foreach ($elements as $element) {
                            $content[] = $element->textContent;
                        }
                        $customContent[$name] = $content;
                    } catch (Exception $e) {
                        $customContent[$name] = ['Error extracting content: ' . $e->getMessage()];
                    }
                }
                
                $result['custom_content'] = $customContent;
            }
            
            $this->logSearch("advancedScrape: Completed advanced scraping of URL: $url");
            return $result;
            
        } catch (Exception $e) {
            $this->logSearch("advancedScrape: Error in advanced scraping of URL: $url", [
                'error' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'url' => $url,
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Process and enhance search results with content analysis
     */
    private function processResults($query, $results) {
        $this->logSearch("processResults: Starting for query='$query'", [
            'result_count' => count($results)
        ]);

        $processedResults = [];
        $lowerQuery = strtolower($query);

        // Generate a comprehensive summary from all results
        if (count($results) > 0) {
            $this->logSearch("processResults: Generating comprehensive summary");
            $summary = $this->generateSummary($query, $results);

            if ($summary) {
                $this->logSearch("processResults: Generated summary", [
                    'summary_length' => strlen($summary['content']),
                    'sources_count' => count($summary['sources'])
                ]);

                // Add the summary as the first result
                array_unshift($processedResults, [
                    'title' => 'Summary: ' . ucfirst($query),
                    'link' => '#summary',
                    'snippet' => $summary['content'],
                    'sources' => $summary['sources']
                ]);
            }
        }

        // Process each result to enhance snippets
        $this->logSearch("processResults: Processing individual results");
        foreach ($results as $index => $result) {
            // Extract the most relevant portion of the content
            $this->logSearch("processResults: Processing result #$index: " . $result['title']);
            $relevantContent = $this->extractRelevantContent($query, $result['content']);

            // Create an enhanced snippet
            $enhancedSnippet = $relevantContent ? $relevantContent : $result['snippet'];
            $this->logSearch("processResults: Enhanced snippet for #$index", [
                'original_length' => strlen($result['snippet']),
                'enhanced_length' => strlen($enhancedSnippet),
                'sample' => substr($enhancedSnippet, 0, 100) . '...'
            ]);

            $processedResults[] = [
                'title' => $result['title'],
                'link' => $result['link'],
                'snippet' => $enhancedSnippet,
                'sources' => isset($result['sources']) ? $result['sources'] : []
            ];
        }

        $this->logSearch("processResults: Complete", [
            'processed_count' => count($processedResults)
        ]);

        return $processedResults;
    }

    /**
     * Generate a comprehensive summary from multiple search results
     */
    private function generateSummary($query, $results) {
        $this->logSearch("generateSummary: Starting summary generation for query: $query");
        // Collect all content and track which sentences come from which sources
        $allSentences = [];
        $sentenceSources = [];
        $sources = [];
        foreach ($results as $index => $result) {
            if (!empty($result['content'])) {
                $sentences = preg_split('/(?<=[.!?])\s+/', $result['content']);
                foreach ($sentences as $sentence) {
                    $allSentences[] = $sentence;
                    $sentenceSources[] = [
                    'title' => $result['title'],
                    'url' => $result['link']
                ];
            }
        }
        }
        if (empty($allSentences)) {
            $this->logSearch("generateSummary: No content to summarize");
            return null;
        }
        // Score sentences for relevance to query
        $scores = [];
        $queryWords = preg_split('/\s+/', strtolower($query));
        foreach ($allSentences as $index => $sentence) {
            $scores[$index] = 0;
            $lowerSentence = strtolower($sentence);
            foreach ($queryWords as $word) {
                if (strlen($word) > 3 && strpos($lowerSentence, $word) !== false) {
                    $scores[$index] += 2;
                }
            }
            if (preg_match('/\b[A-Z][a-z]+ [A-Z][a-z]+\b/', $sentence)) {
                $scores[$index] += 1;
            }
            if (preg_match('/\b\d+\b/', $sentence)) {
                $scores[$index] += 1;
            }
            if (preg_match('/\b(January|February|March|April|May|June|July|August|September|October|November|December)\b/', $sentence)) {
                $scores[$index] += 1;
            }
            $wordCount = str_word_count($sentence);
            if ($wordCount < 5) {
                $scores[$index] -= 2;
            } elseif ($wordCount > 40) {
                $scores[$index] -= 1;
            }
        }
        // Get top sentences
        arsort($scores);
        $topIndices = array_keys(array_slice($scores, 0, 5, true));
        sort($topIndices);
        $summary = '';
        $usedSources = [];
        foreach ($topIndices as $index) {
            if (isset($allSentences[$index]) && strlen($allSentences[$index]) > 20) {
                $summary .= $allSentences[$index] . ' ';
                // Track the source for this sentence
                $src = $sentenceSources[$index];
                $usedSources[$src['url']] = $src; // deduplicate by URL
            }
        }
        $summary = trim($summary);
        // If summary is too short, add more sentences
        if (strlen($summary) < 100 && count($allSentences) > 5) {
            $additionalIndices = array_keys(array_slice($scores, 5, 5, true));
            sort($additionalIndices);
            foreach ($additionalIndices as $index) {
                if (isset($allSentences[$index]) && strlen($allSentences[$index]) > 20) {
                    $summary .= $allSentences[$index] . ' ';
                    $src = $sentenceSources[$index];
                    $usedSources[$src['url']] = $src;
                }
                if (strlen($summary) > 200) {
                    break;
                }
            }
        }
        $this->logSearch("generateSummary: Generated summary", [
            'summary_length' => strlen($summary),
            'sources_count' => count($usedSources)
        ]);
        // Post-process summary to add URLs to source names
        foreach ($usedSources as $src) {
            if (!empty($src['title']) && !empty($src['url'])) {
                $summary = preg_replace(
                    '/\\b' . preg_quote($src['title'], '/') . '\\b/u',
                    $src['title'] . ' (' . $src['url'] . ')',
                    $summary
                );
            }
        }
        return [
            'content' => $summary,
            'sources' => array_values($usedSources)
        ];
    }

    /**
     * Extract presidential information from search results
     */
    private function extractPresidentialInfo($results) {
        $presidentialInfo = null;
        $presidentNamePattern = '/\b(Joe Biden|Kamala Harris|Donald Trump|Barack Obama|George Bush|Bill Clinton)\b/i';

        foreach ($results as $result) {
            // Check if the content mentions a president
            if (preg_match($presidentNamePattern, $result['content'], $matches)) {
                $presidentName = $matches[1];

                // Find sentences mentioning the president
                $sentences = preg_split('/(?<=[.!?])\s+/', $result['content']);
                $relevantSentences = [];

                foreach ($sentences as $sentence) {
                    if (stripos($sentence, $presidentName) !== false) {
                        $relevantSentences[] = $sentence;
                    }
                }

                if (!empty($relevantSentences)) {
                    $presidentialInfo = [
                        'content' => implode(' ', $relevantSentences),
                        'president_name' => $presidentName,
                        'source_title' => $result['title'],
                        'source_url' => $result['link'],
                        'sources' => [
                            [
                                'title' => 'White House',
                                'url' => 'https://whitehouse.gov'
                            ]
                        ]
                    ];

                    return $presidentialInfo;
                }
            }
        }

        return null;
    }

    /**
     * Extract the most relevant content from scraped text based on the query
     */
    private function extractRelevantContent($query, $content) {
        // Split content into sentences
        $sentences = preg_split('/(?<=[.!?])\s+/', $content);

        // Simple relevance scoring based on keyword matching
        $scores = [];
        $queryWords = preg_split('/\s+/', strtolower($query));

        foreach ($sentences as $index => $sentence) {
            $scores[$index] = 0;
            $lowerSentence = strtolower($sentence);

            foreach ($queryWords as $word) {
                if (strlen($word) > 3 && strpos($lowerSentence, $word) !== false) {
                    $scores[$index] += 1;
                }
            }
        }

        // Get the top 3 most relevant sentences
        arsort($scores);
        $topIndices = array_keys(array_slice($scores, 0, 3, true));
        sort($topIndices); // Re-sort by position in the original text

        // Combine the sentences
        $relevantContent = '';
        foreach ($topIndices as $index) {
            if (isset($sentences[$index])) {
                $relevantContent .= $sentences[$index] . ' ';
            }
        }


        return trim($relevantContent);
    }

    /**
     * Generate a cache key from a string
     */
    private function generateCacheKey($str) {
        return md5($str);
    }

    /**
     * Get clean text from the whole document
     */
    private function getCleanText(DomCrawler $crawler) {
        // Clone the crawler to avoid modifying the original
        $clone = clone $crawler;

        // Remove scripts, styles and comments
        $clone->filter('script, style, comment')->each(function (DomCrawler $node) {
            $node->getNode(0)->parentNode->removeChild($node->getNode(0));
        });

        // Get body text
        $text = $clone->filter('body')->text('');

        // Clean up whitespace
        $text = preg_replace('/\s+/', ' ', $text);

        return trim($text);
    }

    /**
     * Summarize text to a certain length
     */
    private function summarizeText($text, $length = 200) {
        if (strlen($text) <= $length) {
            return $text;
        }

        $text = substr($text, 0, $length);
        $pos = strrpos($text, ' ');

        if ($pos !== false) {
            $text = substr($text, 0, $pos);
        }

        return $text . '...';
    }

    /**
     * Convert relative URL to absolute URL
     */
    private function makeAbsoluteUrl($baseUrl, $relativeUrl) {
        // Parse base URL
        $parsedUrl = parse_url($baseUrl);
        $scheme = isset($parsedUrl['scheme']) ? $parsedUrl['scheme'] . '://' : 'https://';
        $host = isset($parsedUrl['host']) ? $parsedUrl['host'] : '';
        $path = isset($parsedUrl['path']) ? $parsedUrl['path'] : '';

        // Handle different relative URL formats
        if (strpos($relativeUrl, '//') === 0) {
            // Protocol-relative URL
            return $scheme . ltrim($relativeUrl, '/');
        } elseif (strpos($relativeUrl, '/') === 0) {
            // Root-relative URL
            return $scheme . $host . $relativeUrl;
        } else {
            // Directory-relative URL
            $dirPath = dirname($path);
            if ($dirPath !== '/') {
                $dirPath .= '/';
            }
            return $scheme . $host . $dirPath . $relativeUrl;
        }
    }

    /**
     * Navigate to a webpage and retrieve console logs
     *
     * @param string $url The URL to navigate to
     * @return array The page information and console logs
     */
    public function getConsoleLogs($url) {
        $this->logSearch("getConsoleLogs: Navigating to URL: $url and retrieving console logs");

        try {
            $pantherClient = PantherClient::createChromeClient([
                'arguments' => [
                    '--disable-web-security',
                    '--allow-running-insecure-content',
                    '--disable-extensions',
                    '--no-sandbox',
                    '--disable-dev-shm-usage'
                ]
            ]);

            $browser = $pantherClient->request('GET', $url);

            // Wait for page to load
            $pantherClient->waitFor('#app, body, html', 10);

            // Get page information
            $title = $browser->findElement(WebDriverBy::tagName('title'))->getText();
            $currentUrl = $pantherClient->getCurrentURL();

            // Get console logs
            $consoleLogs = $pantherClient->getWebDriver()->manage()->getLog('browser');

            $result = [
                'success' => true,
                'url' => $url,
                'current_url' => $currentUrl,
                'title' => $title,
                'console_logs' => $consoleLogs,
                'timestamp' => date('Y-m-d H:i:s')
            ];

            $this->logSearch("getConsoleLogs: Successfully retrieved console logs from URL: $url");
            return $result;

        } catch (Exception $e) {
            $this->logSearch("getConsoleLogs: Error retrieving console logs from URL: $url", [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'url' => $url,
                'console_logs' => [],
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Navigate to a webpage and execute custom JavaScript
     *
     * @param string $url The URL to navigate to
     * @param string $javascript The JavaScript code to execute
     * @return array The result of JavaScript execution
     */
    public function executeJavaScript($url, $javascript) {
        $this->logSearch("executeJavaScript: Navigating to URL: $url and executing JavaScript");

        try {
            $pantherClient = PantherClient::createChromeClient();

            $browser = $pantherClient->request('GET', $url);

            // Wait for page to load
            $pantherClient->waitFor('#app, body, html', 10);

            // Execute JavaScript
            $result = $pantherClient->executeScript($javascript);

            $response = [
                'success' => true,
                'url' => $url,
                'javascript' => $javascript,
                'result' => $result,
                'timestamp' => date('Y-m-d H:i:s')
            ];

            $this->logSearch("executeJavaScript: Successfully executed JavaScript on URL: $url");
            return $response;

        } catch (Exception $e) {
            $this->logSearch("executeJavaScript: Error executing JavaScript on URL: $url", [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'url' => $url,
                'javascript' => $javascript,
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Interact with webpage elements (click, fill forms, etc.)
     *
     * @param string $url The URL to navigate to
     * @param array $interactions Array of interactions to perform
     * @return array The result of the interactions
     */
    public function interactWithPage($url, $interactions) {
        $this->logSearch("interactWithPage: Navigating to URL: $url and performing interactions");

        try {
            $pantherClient = PantherClient::createChromeClient();

            $browser = $pantherClient->request('GET', $url);

            // Wait for page to load
            $pantherClient->waitFor('#app, body, html', 10);

            $interactionResults = [];

            foreach ($interactions as $interaction) {
                $type = $interaction['type'] ?? '';
                $selector = $interaction['selector'] ?? '';

                try {
                    switch ($type) {
                        case 'click':
                            $element = $browser->findElement(WebDriverBy::cssSelector($selector));
                            $element->click();
                            $interactionResults[] = [
                                'type' => $type,
                                'selector' => $selector,
                                'status' => 'success',
                                'message' => "Element clicked successfully"
                            ];
                            break;

                        case 'fill':
                            $element = $browser->findElement(WebDriverBy::cssSelector($selector));
                            $element->clear();
                            $element->sendKeys($interaction['value'] ?? '');
                            $interactionResults[] = [
                                'type' => $type,
                                'selector' => $selector,
                                'value' => $interaction['value'] ?? '',
                                'status' => 'success',
                                'message' => "Value filled successfully"
                            ];
                            break;

                        case 'get_text':
                            $element = $browser->findElement(WebDriverBy::cssSelector($selector));
                            $text = $element->getText();
                            $interactionResults[] = [
                                'type' => $type,
                                'selector' => $selector,
                                'text' => $text,
                                'status' => 'success',
                                'message' => "Text retrieved successfully"
                            ];
                            break;

                        default:
                            $interactionResults[] = [
                                'type' => $type,
                                'selector' => $selector,
                                'status' => 'error',
                                'message' => "Unknown interaction type: $type"
                            ];
                    }
                } catch (Exception $e) {
                    $interactionResults[] = [
                        'type' => $type,
                        'selector' => $selector,
                        'status' => 'error',
                        'message' => $e->getMessage()
                    ];
                }
            }

            $result = [
                'success' => true,
                'url' => $url,
                'interactions' => $interactionResults,
                'timestamp' => date('Y-m-d H:i:s')
            ];

            $this->logSearch("interactWithPage: Successfully performed interactions on URL: $url");
            return $result;

        } catch (Exception $e) {
            $this->logSearch("interactWithPage: Error performing interactions on URL: $url", [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'url' => $url,
                'interactions' => [],
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Get performance metrics for a webpage
     *
     * @param string $url The URL to analyze
     * @return array Performance metrics
     */
    public function getPerformanceMetrics($url) {
        $this->logSearch("getPerformanceMetrics: Getting performance metrics for URL: $url");

        try {
            $pantherClient = PantherClient::createChromeClient();

            $browser = $pantherClient->request('GET', $url);

            // Wait for page to load completely
            $pantherClient->waitFor('#app, body, html', 10);

            // Get performance timing data using JavaScript
            $timing = $pantherClient->executeScript("
                var timing = performance.timing;
                return {
                    'navigation_start': timing.navigationStart,
                    'unload_event_start': timing.unloadEventStart,
                    'unload_event_end': timing.unloadEventEnd,
                    'redirect_start': timing.redirectStart,
                    'redirect_end': timing.redirectEnd,
                    'fetch_start': timing.fetchStart,
                    'domain_lookup_start': timing.domainLookupStart,
                    'domain_lookup_end': timing.domainLookupEnd,
                    'connect_start': timing.connectStart,
                    'connect_end': timing.connectEnd,
                    'secure_connection_start': timing.secureConnectionStart,
                    'request_start': timing.requestStart,
                    'response_start': timing.responseStart,
                    'response_end': timing.responseEnd,
                    'dom_loading': timing.domLoading,
                    'dom_interactive': timing.domInteractive,
                    'dom_content_loaded_event_start': timing.domContentLoadedEventStart,
                    'dom_content_loaded_event_end': timing.domContentLoadedEventEnd,
                    'dom_complete': timing.domComplete,
                    'load_event_start': timing.loadEventStart,
                    'load_event_end': timing.loadEventEnd
                };
            ");

            // Calculate derived metrics
            $calculatedMetrics = [
                'page_load_time' => $timing['load_event_end'] - $timing['navigation_start'],
                'dom_ready_time' => $timing['dom_content_loaded_event_end'] - $timing['navigation_start'],
                'time_to_first_byte' => $timing['response_start'] - $timing['navigation_start'],
                'connection_time' => $timing['connect_end'] - $timing['connect_start'],
                'dns_lookup_time' => $timing['domain_lookup_end'] - $timing['domain_lookup_start'],
                'redirect_time' => $timing['redirect_end'] - $timing['redirect_start']
            ];

            // Get resource information
            $resources = $pantherClient->executeScript("
                var resources = performance.getEntriesByType('resource');
                var resourceList = [];
                for (var i = 0; i < resources.length; i++) {
                    resourceList.push({
                        'name': resources[i].name,
                        'entryType': resources[i].entryType,
                        'startTime': resources[i].startTime,
                        'duration': resources[i].duration,
                        'size': resources[i].transferSize || 0
                    });
                }
                return resourceList;
            ");

            $result = [
                'success' => true,
                'url' => $url,
                'timing' => $timing,
                'calculated_metrics' => $calculatedMetrics,
                'resources' => $resources,
                'timestamp' => date('Y-m-d H:i:s')
            ];

            $this->logSearch("getPerformanceMetrics: Successfully retrieved performance metrics for URL: $url");
            return $result;

        } catch (Exception $e) {
            $this->logSearch("getPerformanceMetrics: Error retrieving performance metrics for URL: $url", [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'url' => $url,
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Execute a multi-step workflow
     *
     * @param array $steps Array of workflow steps to execute
     * @return array Results of the workflow execution
     */
    public function executeWorkflow($steps) {
        $this->logSearch("executeWorkflow: Starting workflow with " . count($steps) . " steps");

        $results = [];
        $sessionData = []; // Store data between steps (like cookies, authentication tokens, etc.)

        try {
            $pantherClient = PantherClient::createChromeClient();

            foreach ($steps as $index => $step) {
                $action = $step['action'] ?? '';
                $url = $step['url'] ?? '';

                $this->logSearch("executeWorkflow: Executing step $index: $action for URL: $url");

                $stepResult = null;

                try {
                    switch ($action) {
                        case 'navigate':
                            $browser = $pantherClient->request('GET', $url);
                            $pantherClient->waitFor('#app, body, html', 10);

                            $stepResult = [
                                'step' => $index,
                                'action' => $action,
                                'url' => $url,
                                'status' => 'success',
                                'message' => 'Navigation successful',
                                'current_url' => $pantherClient->getCurrentURL(),
                                'title' => $browser->findElement(WebDriverBy::tagName('title'))->getText()
                            ];
                            break;

                        case 'interact':
                            $interactions = $step['interactions'] ?? [];
                            $stepResult = $this->interactWithPage($url, $interactions);
                            $stepResult['step'] = $index;
                            $stepResult['action'] = $action;
                            break;

                        case 'extract':
                            $selectors = $step['selectors'] ?? [];
                            $stepResult = $this->advancedScrape($url, $selectors);
                            $stepResult['step'] = $index;
                            $stepResult['action'] = $action;
                            break;

                        case 'wait':
                            $waitTime = $step['time'] ?? 1; // Default to 1 second
                            sleep($waitTime);
                            $stepResult = [
                                'step' => $index,
                                'action' => $action,
                                'time' => $waitTime,
                                'status' => 'success',
                                'message' => "Waited for {$waitTime} seconds"
                            ];
                            break;

                        case 'form_submit':
                            $formData = $step['form_data'] ?? [];
                            $formSelector = $step['form_selector'] ?? 'form';

                            $browser = $pantherClient->request('GET', $url);
                            $pantherClient->waitFor($formSelector, 5);

                            // Fill form fields
                            foreach ($formData as $field => $value) {
                                $input = $browser->findElement(WebDriverBy::cssSelector("[name='$field']"));
                                $input->clear();
                                $input->sendKeys($value);
                            }

                            // Submit the form
                            $form = $browser->findElement(WebDriverBy::cssSelector($formSelector));
                            $form->submit();

                            $stepResult = [
                                'step' => $index,
                                'action' => $action,
                                'form_data' => $formData,
                                'status' => 'success',
                                'message' => 'Form submitted successfully',
                                'current_url' => $pantherClient->getCurrentURL()
                            ];
                            break;

                        default:
                            $stepResult = [
                                'step' => $index,
                                'action' => $action,
                                'status' => 'error',
                                'message' => "Unknown action: $action"
                            ];
                    }
                } catch (Exception $e) {
                    $stepResult = [
                        'step' => $index,
                        'action' => $action,
                        'status' => 'error',
                        'message' => $e->getMessage()
                    ];
                }

                $results[] = $stepResult;

                // If any step fails, we might want to stop the workflow
                if (isset($stepResult['status']) && $stepResult['status'] === 'error') {
                    if (isset($step['fail_on_error']) && $step['fail_on_error'] === false) {
                        // Continue despite error
                    } else {
                        // Stop the workflow
                        break;
                    }
                }
            }

            $workflowResult = [
                'success' => true,
                'steps_executed' => count($results),
                'results' => $results,
                'timestamp' => date('Y-m-d H:i:s')
            ];

            $this->logSearch("executeWorkflow: Successfully completed workflow with " . count($results) . " steps");
            return $workflowResult;

        } catch (Exception $e) {
            $this->logSearch("executeWorkflow: Error executing workflow", [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'error' => $e->getMessage(),
                'results' => $results, // Include partial results
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Submit a form on a webpage
     *
     * @param string $url The URL containing the form
     * @param array $formData The form data to submit
     * @param string $formSelector The CSS selector for the form (default: 'form')
     * @return array Result of the form submission
     */
    public function submitForm($url, $formData, $formSelector = 'form') {
        $this->logSearch("submitForm: Submitting form at URL: $url");

        try {
            $pantherClient = PantherClient::createChromeClient();

            $browser = $pantherClient->request('GET', $url);

            // Wait for form to load
            $pantherClient->waitFor($formSelector, 10);

            // Fill form fields
            foreach ($formData as $field => $value) {
                try {
                    // Try to find the input by name attribute first
                    $input = $browser->findElement(WebDriverBy::cssSelector("[name='$field']"));
                    $input->clear();
                    $input->sendKeys($value);
                } catch (Exception $e) {
                    // If not found by name, try by ID
                    try {
                        $input = $browser->findElement(WebDriverBy::cssSelector("#$field"));
                        $input->clear();
                        $input->sendKeys($value);
                    } catch (Exception $e2) {
                        // If not found by name or ID, log the error
                        $this->logSearch("submitForm: Could not find field '$field' in form", [
                            'error' => $e2->getMessage()
                        ]);
                    }
                }
            }

            // Submit the form
            $form = $browser->findElement(WebDriverBy::cssSelector($formSelector));
            $form->submit();

            // Wait for the page to load after submission
            $pantherClient->waitFor('#app, body, html', 10);

            $result = [
                'success' => true,
                'url' => $url,
                'form_data' => $formData,
                'form_selector' => $formSelector,
                'current_url' => $pantherClient->getCurrentURL(),
                'message' => 'Form submitted successfully',
                'timestamp' => date('Y-m-d H:i:s')
            ];

            $this->logSearch("submitForm: Successfully submitted form at URL: $url");
            return $result;

        } catch (Exception $e) {
            $this->logSearch("submitForm: Error submitting form at URL: $url", [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'url' => $url,
                'form_data' => $formData,
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Manage cookies for a webpage
     *
     * @param string $url The URL to manage cookies for
     * @param string $operation The operation to perform: 'get', 'set', 'delete', 'clear'
     * @param array $cookieData Cookie data for set operation
     * @return array Result of the cookie operation
     */
    public function manageCookies($url, $operation = 'get', $cookieData = []) {
        $this->logSearch("manageCookies: Performing '$operation' operation on cookies for URL: $url");

        try {
            $pantherClient = PantherClient::createChromeClient();

            $browser = $pantherClient->request('GET', $url);

            // Wait for page to load
            $pantherClient->waitFor('#app, body, html', 5);

            $result = [
                'success' => true,
                'url' => $url,
                'operation' => $operation,
                'timestamp' => date('Y-m-d H:i:s')
            ];

            switch ($operation) {
                case 'get':
                    $cookies = $pantherClient->getWebDriver()->manage()->getCookies();
                    $result['cookies'] = $cookies;
                    $result['message'] = count($cookies) . ' cookies retrieved';
                    break;

                case 'set':
                    foreach ($cookieData as $cookie) {
                        // Ensure required fields are present
                        if (!isset($cookie['name']) || !isset($cookie['value'])) {
                            continue;
                        }

                        $cookieArray = [
                            'name' => $cookie['name'],
                            'value' => $cookie['value'],
                            'path' => $cookie['path'] ?? '/',
                            'domain' => $cookie['domain'] ?? parse_url($url, PHP_URL_HOST),
                        ];

                        if (isset($cookie['secure'])) {
                            $cookieArray['secure'] = $cookie['secure'];
                        }

                        if (isset($cookie['httpOnly'])) {
                            $cookieArray['httpOnly'] = $cookie['httpOnly'];
                        }

                        $pantherClient->getWebDriver()->manage()->addCookie($cookieArray);
                    }
                    $result['message'] = 'Cookies set successfully';
                    break;

                case 'delete':
                    if (isset($cookieData['name'])) {
                        $pantherClient->getWebDriver()->manage()->deleteCookieNamed($cookieData['name']);
                        $result['message'] = 'Cookie deleted successfully';
                    } else {
                        $result['success'] = false;
                        $result['message'] = 'Cookie name required for delete operation';
                    }
                    break;

                case 'clear':
                    $pantherClient->getWebDriver()->manage()->deleteAllCookies();
                    $result['message'] = 'All cookies cleared';
                    break;

                default:
                    $result['success'] = false;
                    $result['message'] = "Invalid operation: $operation. Use 'get', 'set', 'delete', or 'clear'";
            }

            $this->logSearch("manageCookies: Successfully completed '$operation' operation for URL: $url");
            return $result;

        } catch (Exception $e) {
            $this->logSearch("manageCookies: Error performing '$operation' operation for URL: $url", [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'url' => $url,
                'operation' => $operation,
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Login to a website using username and password
     *
     * @param string $url The login page URL
     * @param string $username The username
     * @param string $password The password
     * @param array $selectors Selectors for login form fields (optional)
     * @return array Result of the login operation
     */
    public function login($url, $username, $password, $selectors = []) {
        $this->logSearch("login: Attempting login to URL: $url with username: $username");

        // Default selectors
        $defaultSelectors = [
            'username_field' => 'input[name="username"], input[name="email"], input[type="email"], #username, #email',
            'password_field' => 'input[type="password"], #password',
            'submit_button' => 'button[type="submit"], input[type="submit"], #login-btn'
        ];

        $selectors = array_merge($defaultSelectors, $selectors);

        try {
            $pantherClient = PantherClient::createChromeClient();

            $browser = $pantherClient->request('GET', $url);

            // Wait for login form to load
            $pantherClient->waitFor($selectors['username_field'], 10);

            // Fill username
            $usernameField = $browser->findElement(WebDriverBy::cssSelector($selectors['username_field']));
            $usernameField->clear();
            $usernameField->sendKeys($username);

            // Fill password
            $passwordField = $browser->findElement(WebDriverBy::cssSelector($selectors['password_field']));
            $passwordField->clear();
            $passwordField->sendKeys($password);

            // Click submit button
            $submitBtn = $browser->findElement(WebDriverBy::cssSelector($selectors['submit_button']));
            $submitBtn->click();

            // Wait for page to load after login
            sleep(2); // Give time for potential redirects

            // Check if we're still on the login page (indicating failed login)
            $currentUrl = $pantherClient->getCurrentURL();
            $isLoginPage = strpos($currentUrl, 'login') !== false || strpos($currentUrl, 'signin') !== false;

            $result = [
                'success' => !$isLoginPage, // Success if we're not on login page anymore
                'url' => $url,
                'username' => $username,
                'current_url' => $currentUrl,
                'message' => $isLoginPage ? 'Login failed - still on login page' : 'Login successful',
                'timestamp' => date('Y-m-d H:i:s')
            ];

            $this->logSearch("login: Login attempt " . ($result['success'] ? 'successful' : 'failed') . " for URL: $url");
            return $result;

        } catch (Exception $e) {
            $this->logSearch("login: Error during login to URL: $url", [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'url' => $url,
                'username' => $username,
                'error' => $e->getMessage(),
                'message' => 'Login failed due to error',
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Wait for specific conditions on a webpage
     *
     * @param string $url The URL to navigate to
     * @param array $conditions Array of conditions to wait for
     * @param int $timeout Maximum time to wait in seconds
     * @return array Result of the wait operation
     */
    public function waitForConditions($url, $conditions, $timeout = 10) {
        $this->logSearch("waitForConditions: Waiting for conditions on URL: $url");

        try {
            $pantherClient = PantherClient::createChromeClient();

            $browser = $pantherClient->request('GET', $url);

            $results = [];

            foreach ($conditions as $condition) {
                $type = $condition['type'] ?? '';
                $selector = $condition['selector'] ?? '';
                $text = $condition['text'] ?? '';

                $conditionResult = [
                    'type' => $type,
                    'selector' => $selector,
                    'status' => 'pending'
                ];

                try {
                    switch ($type) {
                        case 'element_present':
                            $pantherClient->waitFor($selector, $timeout);
                            $conditionResult['status'] = 'success';
                            $conditionResult['message'] = "Element '$selector' is present";
                            break;

                        case 'element_visible':
                            $pantherClient->waitForVisibility($selector, $timeout);
                            $conditionResult['status'] = 'success';
                            $conditionResult['message'] = "Element '$selector' is visible";
                            break;

                        case 'text_present':
                            $pantherClient->waitForXPath("//*[contains(text(), '$text')]", $timeout);
                            $conditionResult['status'] = 'success';
                            $conditionResult['message'] = "Text '$text' is present";
                            break;

                        case 'url_contains':
                            $startTime = time();
                            while ((time() - $startTime) < $timeout) {
                                $currentUrl = $pantherClient->getCurrentURL();
                                if (strpos($currentUrl, $text) !== false) {
                                    $conditionResult['status'] = 'success';
                                    $conditionResult['message'] = "URL contains '$text'";
                                    $conditionResult['current_url'] = $currentUrl;
                                    break;
                                }
                                sleep(1);
                            }
                            if ($conditionResult['status'] !== 'success') {
                                $conditionResult['status'] = 'timeout';
                                $conditionResult['message'] = "Timeout waiting for URL to contain '$text'";
                            }
                            break;

                        default:
                            $conditionResult['status'] = 'error';
                            $conditionResult['message'] = "Unknown condition type: $type";
                    }
                } catch (Exception $e) {
                    $conditionResult['status'] = 'error';
                    $conditionResult['message'] = $e->getMessage();
                }

                $results[] = $conditionResult;
            }

            $allSuccessful = true;
            foreach ($results as $result) {
                if ($result['status'] !== 'success') {
                    $allSuccessful = false;
                    break;
                }
            }

            $result = [
                'success' => $allSuccessful,
                'url' => $url,
                'conditions' => $results,
                'timestamp' => date('Y-m-d H:i:s')
            ];

            $this->logSearch("waitForConditions: Completed waiting for conditions on URL: $url");
            return $result;

        } catch (Exception $e) {
            $this->logSearch("waitForConditions: Error waiting for conditions on URL: $url", [
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'url' => $url,
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }

    /**
     * Flatten a scraped result tree into an array of attachments (screenshot+text)
     * @param array $result
     * @return array
     */
    public static function flattenScrapedResults($result) {
        $attachments = [];
        if (!empty($result['screenshot_base64']) && !empty($result['content'])) {
            $attachments[] = [
                'base64' => $result['screenshot_base64'],
                'text' => $result['content'],
                'url' => $result['url'],
                'title' => $result['title']
            ];
        }
        if (!empty($result['children']) && is_array($result['children'])) {
            foreach ($result['children'] as $child) {
                $attachments = array_merge($attachments, self::flattenScrapedResults($child));
            }
        }
        return $attachments;
    }
}
?>