<?php

// Require Composer autoload
if (file_exists('../vendor/autoload.php')) {
    require_once '../vendor/autoload.php';
} elseif (file_exists('vendor/autoload.php')) {
    require_once 'vendor/autoload.php';
} else {
    throw new Exception('Composer autoload not found. Please run "composer install" first.');
}

use Goutte\Client as GoutteClient;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Component\DomCrawler\Crawler as DomCrawler;
use Symfony\Component\Panther\Client as PantherClient;

/**
 * WebSearchManager Class
 * Handles web searches and content scraping
 */
class WebSearchManager {
    private $client;
    private $cacheDir;
    private $cacheExpiration; // in seconds
    
    /**
     * Class constants
     */
    private $searchApiKey;
    private $searchEngineId; 
    
    public function __construct($cacheDir = '../cache/search', $cacheExpiration = 3600) {
        $this->client = new GoutteClient(HttpClient::create([
            'headers' => [
                'User-Agent' => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/91.0.4472.124 Safari/537.36',
                'Accept-Language' => 'en-US,en;q=0.9',
            ],
            'timeout' => 30,
        ]));
        
        $this->cacheDir = rtrim($cacheDir, '/');
        $this->cacheExpiration = $cacheExpiration;
        
        // Set Google Custom Search API credentials
        $this->searchApiKey = "AIzaSyCPR3fFs89C3KoUqlfQ3D7fuY_qE6QMtq8";
        $this->searchEngineId = "b5c87db0a2743426b";
        
        // Create cache directory if it doesn't exist
        if (!file_exists($this->cacheDir)) {
            mkdir($this->cacheDir, 0755, true);
        }
    }
    
    /**
     * Log a message to the search log file
     * 
     * @param string $message The message to log
     * @param mixed $data Optional data to include
     */
    private function logSearch($message, $data = null) {
        $logDir = '../logs';
        if (!file_exists($logDir)) {
            mkdir($logDir, 0755, true);
        }
        
        $logFile = $logDir . '/search_log.txt';
        $timestamp = date('Y-m-d H:i:s');
        $logMessage = "[$timestamp] $message";
        
        if ($data !== null) {
            $logMessage .= "\nData: " . json_encode($data, JSON_PRETTY_PRINT);
        }
        
        $logMessage .= "\n" . str_repeat('-', 80) . "\n";
        
        // Append to log file
        file_put_contents($logFile, $logMessage, FILE_APPEND);
    }
    
    /**
     * Search the web for information
     * 
     * @param string $query The search query
     * @param int $limit Maximum number of results to return
     * @return array Search results
     */
    public function search($query, $limit = 5) {
        $this->logSearch("SEARCH START: Query='$query', Limit=$limit");
        
        try {
            // Step 1: Get search results from Google Custom Search API
            $this->logSearch("Step 1: Getting initial search results from Google API");
            
            $searchResults = $this->getGoogleApiResults($query, $limit);
            
            $this->logSearch("Step 1 Complete: Initial results obtained", [
                'count' => count($searchResults['results']),
                'sample' => array_slice($searchResults['results'], 0, 2)
            ]);
            
            // Step 2: Scrape content from top results
            $this->logSearch("Step 2: Scraping content from top results");
            $enhancedResults = [];
            foreach ($searchResults['results'] as $index => $result) {
                if ($index >= 5) break; // Limit to top 5 for scraping
                
                // Get the URL
                $url = $result['link'];
                $this->logSearch("Step 2.$index: Scraping URL: $url");
                
                try {
                    // Scrape the page
                    $scrapedContent = $this->scrapeUrl($url);
                    
                    $contentLength = strlen($scrapedContent['content'] ?? '');
                    $this->logSearch("Step 2.$index: Scraping completed for URL: $url", [
                        'content_length' => $contentLength,
                        'success' => $scrapedContent['success'],
                        'title' => $scrapedContent['title']
                    ]);
                    
                    // Save the relevant content
                    $enhancedResults[] = [
                        'title' => $result['title'],
                        'link' => $url,
                        'snippet' => $result['snippet'],
                        'content' => $scrapedContent['content'] ?? 'No content available',
                        'html_content' => $scrapedContent['html_content'] ?? '',
                        'footer_content' => $scrapedContent['footer_content'] ?? ''
                    ];
                } catch (Exception $e) {
                    $this->logSearch("Step 2.$index: Scraping failed for URL: $url", [
                        'error' => $e->getMessage()
                    ]);
                    
                    // Still add the result but without enhanced content
                    $enhancedResults[] = [
                        'title' => $result['title'],
                        'link' => $url,
                        'snippet' => $result['snippet'],
                        'content' => $result['snippet'], // Fall back to snippet
                        'html_content' => '<html><body>' . $result['snippet'] . '</body></html>', // Basic HTML wrapper around snippet
                        'footer_content' => '' // No footer content available
                    ];
                }
            }
            
            // Step 3: Process and summarize the content
            $this->logSearch("Step 3: Processing and summarizing content");
            $processedResults = $this->processResults($query, $enhancedResults);
            $this->logSearch("Step 3 Complete: Results processed", [
                'processed_count' => count($processedResults),
                'sample' => array_slice($processedResults, 0, 2)
            ]);
            
            // Step 4: Return the enhanced search results
            $finalResult = [
                'success' => true,
                'query' => $query,
                'results' => $processedResults,
                'timestamp' => date('Y-m-d H:i:s')
            ];
            
            $this->logSearch("SEARCH COMPLETE: Query='$query'", [
                'status' => 'success',
                'result_count' => count($processedResults)
            ]);
            
            return $finalResult;
        } catch (Exception $e) {
            $errorMsg = "SEARCH ERROR: " . $e->getMessage();
            $this->logSearch($errorMsg, [
                'query' => $query,
                'trace' => $e->getTraceAsString()
            ]);
            
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'query' => $query
            ];
        }
    }
    
    /**
     * Get search results from Google Custom Search API
     * 
     * @param string $query The search query
     * @param int $limit Maximum number of results to return
     * @return array Search results from Google API
     */
    private function getGoogleApiResults($query, $limit = 5) {
        $this->logSearch("getGoogleApiResults: Searching Google for '$query'");
        
        try {
            // Construct the Google Custom Search API URL
            $apiUrl = "https://www.googleapis.com/customsearch/v1?key={$this->searchApiKey}&cx={$this->searchEngineId}&q=" . urlencode($query) . "&num={$limit}";
            
            $this->logSearch("getGoogleApiResults: Using API URL: $apiUrl");
            
            // Make the HTTP request to Google API
            $response = file_get_contents($apiUrl);
            
            if ($response === false) {
                throw new Exception("Failed to get response from Google API");
            }
            
            // Parse the JSON response
            $data = json_decode($response, true);
            
            if (json_last_error() !== JSON_ERROR_NONE) {
                throw new Exception("Failed to parse JSON response: " . json_last_error_msg());
            }
            
            $this->logSearch("getGoogleApiResults: API response parsed successfully");
            
            // Extract search results
            $results = [];
            
            if (isset($data['items']) && is_array($data['items'])) {
                foreach ($data['items'] as $item) {
                    $results[] = [
                        'title' => $item['title'] ?? 'No title',
                        'link' => $item['link'] ?? '',
                        'snippet' => $item['snippet'] ?? '',
                        'source' => 'google_api'
                    ];
                }
            }
            
            $this->logSearch("getGoogleApiResults: Extracted " . count($results) . " results");
            
            // For president-specific queries, we can try to provide more accurate information
            if (count($results) == 0 && 
                (strpos(strtolower($query), 'president') !== false && 
                 (strpos(strtolower($query), 'usa') !== false || 
                  strpos(strtolower($query), 'united states') !== false || 
                  strpos(strtolower($query), 'america') !== false))) {
                
                
            }
            
            // Format and return the results
            return [
                'success' => true,
                'query' => $query,
                'results' => array_slice($results, 0, $limit),
                'source' => 'google_api',
                'timestamp' => date('Y-m-d H:i:s')
            ];

        } catch (Exception $e) {
            $this->logSearch("getGoogleApiResults: Error occurred while searching Google API", [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            // Return error result instead of mock data
            return [
                'success' => false,
                'query' => $query,
                'results' => [],
                'error' => $e->getMessage(),
                'timestamp' => date('Y-m-d H:i:s')
            ];
        }
    }
    
    /**
     * Scrape content from a URL (now supports JS-rendered pages, bot/captcha detection, screenshots, recursion)
     *
     * @param string $url The URL to scrape
     * @param int $depth Recursion depth (default 1)
     * @param array $visited Internal: visited URLs
     * @return array The scraped content
     */
    public function scrapeUrl($url, $depth = 1, &$visited = []) {
        $result = [
            'url' => $url,
            'success' => false,
            'title' => '',
            'description' => '',
            'content' => '',
            'footer_content' => '',
            'html_content' => '',
            'timestamp' => date('Y-m-d H:i:s'),
            'screenshot_base64' => '',
            'bot_detection' => [
                'recaptcha' => false,
                'js_challenge' => false,
                'cookie_challenge' => false,
                'other' => false
            ],
            'links' => [],
            'children' => []
        ];
        $visited[$url] = true;
        try {
            // Try Goutte first for speed
            try {
                $crawler = $this->client->request('GET', $url);
                $title = $crawler->filter('title')->text('');
                if (empty($title)) {
                    $title = $crawler->filter('h1')->text('');
                }
                $description = '';
                $crawler->filter('meta[name="description"]')->each(function (DomCrawler $node) use (&$description) {
                    $description = $node->attr('content');
                });
                $mainContent = $this->getCleanText($crawler);
                $footerContent = '';
                $footerSelectors = ['footer', '.footer', '#footer', '.site-footer', '.page-footer', '.main-footer', '.global-footer'];
                foreach ($footerSelectors as $selector) {
                    if ($crawler->filter($selector)->count() > 0) {
                        $footerContent = $crawler->filter($selector)->text();
                        break;
                    }
                }
                $fullHtml = '';
                try {
                    $html = $crawler->html();
                    if (!empty($html)) {
                        $fullHtml = $html;
                    }
                } catch (Exception $e) {}
                $result['success'] = true;
                $result['title'] = $title;
                $result['description'] = $description;
                $result['content'] = trim(preg_replace('/\s+/', ' ', $mainContent));
                $result['footer_content'] = trim(preg_replace('/\s+/', ' ', $footerContent));
                $result['html_content'] = $fullHtml;
                // Extract links
                $result['links'] = $crawler->filter('a')->each(function (DomCrawler $node) use ($url) {
                    $href = $node->attr('href');
                    if ($href) {
                        return $this->makeAbsoluteUrl($url, $href);
                    }
                    return null;
                });
                $result['links'] = array_filter($result['links']);
            } catch (Exception $e) {
                // Goutte failed (maybe JS site), try Panther
                $pantherClient = PantherClient::createChromeClient();
                $browser = $pantherClient->request('GET', $url);
                // Wait for JS rendering (wait for body)
                $pantherClient->waitFor('#app, body, html', 10);
                $title = $browser->filter('title')->text('');
                if (empty($title)) {
                    $title = $browser->filter('h1')->text('');
                }
                $description = '';
                $browser->filter('meta[name="description"]')->each(function (DomCrawler $node) use (&$description) {
                    $description = $node->attr('content');
                });
                $mainContent = $this->getCleanText($browser);
                $footerContent = '';
                $footerSelectors = ['footer', '.footer', '#footer', '.site-footer', '.page-footer', '.main-footer', '.global-footer'];
                foreach ($footerSelectors as $selector) {
                    if ($browser->filter($selector)->count() > 0) {
                        $footerContent = $browser->filter($selector)->text();
                        break;
                    }
                }
                $fullHtml = '';
                try {
                    $html = $browser->html();
                    if (!empty($html)) {
                        $fullHtml = $html;
                    }
                } catch (Exception $e) {}
                // Screenshot
                try {
                    $screenshot = $pantherClient->takeScreenshot();
                    $result['screenshot_base64'] = base64_encode($screenshot);
                } catch (Exception $e) {
                    $result['screenshot_base64'] = '';
                }
                // Bot/captcha detection
                $htmlLower = strtolower($fullHtml);
                if (strpos($htmlLower, 'recaptcha') !== false || strpos($htmlLower, 'g-recaptcha') !== false) {
                    $result['bot_detection']['recaptcha'] = true;
                }
                if (strpos($htmlLower, 'cf-chl-captcha') !== false || strpos($htmlLower, 'js-challenge') !== false) {
                    $result['bot_detection']['js_challenge'] = true;
                }
                if (strpos($htmlLower, 'cookie') !== false && strpos($htmlLower, 'consent') !== false) {
                    $result['bot_detection']['cookie_challenge'] = true;
                }
                $result['success'] = true;
                $result['title'] = $title;
                $result['description'] = $description;
                $result['content'] = trim(preg_replace('/\s+/', ' ', $mainContent));
                $result['footer_content'] = trim(preg_replace('/\s+/', ' ', $footerContent));
                $result['html_content'] = $fullHtml;
                // Extract links
                $result['links'] = $browser->filter('a')->each(function (DomCrawler $node) use ($url) {
                    $href = $node->attr('href');
                    if ($href) {
                        return $this->makeAbsoluteUrl($url, $href);
                    }
                    return null;
                });
                $result['links'] = array_filter($result['links']);
            }
            // Recursion (scrape links)
            if ($depth > 1) {
                $maxChildren = 5; // Limit recursion for performance
                $children = [];
                $count = 0;
                foreach ($result['links'] as $link) {
                    if (!isset($visited[$link]) && $count < $maxChildren) {
                        $children[] = $this->scrapeUrl($link, $depth - 1, $visited);
                        $count++;
                    }
                }
                $result['children'] = $children;
            }
        } catch (Exception $e) {
            $this->logSearch("scrapeUrl: Error scraping URL: $url", [
                'error' => $e->getMessage()
            ]);
            // Return error result instead of mock data
            return [
                'url' => $url,
                'success' => false,
                'title' => 'Error scraping page',
                'description' => '',
                'content' => 'Failed to scrape content from the URL: ' . $e->getMessage(),
                'footer_content' => '',
                'html_content' => '',
                'timestamp' => date('Y-m-d H:i:s'),
                'screenshot_base64' => '',
                'bot_detection' => [
                    'recaptcha' => false,
                    'js_challenge' => false,
                    'cookie_challenge' => false,
                    'other' => false
                ],
                'links' => [],
                'children' => []
            ];
        }
        return $result;
    }
    
    /**
     * Process and enhance search results with content analysis
     */
    private function processResults($query, $results) {
        $this->logSearch("processResults: Starting for query='$query'", [
            'result_count' => count($results)
        ]);
        
        $processedResults = [];
        $lowerQuery = strtolower($query);
        
        
        // Generate a comprehensive summary from all results
        if (count($results) > 0) {
            $this->logSearch("processResults: Generating comprehensive summary");
            $summary = $this->generateSummary($query, $results);
            
            if ($summary) {
                $this->logSearch("processResults: Generated summary", [
                    'summary_length' => strlen($summary['content']),
                    'sources_count' => count($summary['sources'])
                ]);
                
                // Add the summary as the first result
                array_unshift($processedResults, [
                    'title' => 'Summary: ' . ucfirst($query),
                    'link' => '#summary',
                    'snippet' => $summary['content'],
                    'sources' => $summary['sources']
                ]);
            }
        }
        
        // Process each result to enhance snippets
        $this->logSearch("processResults: Processing individual results");
        foreach ($results as $index => $result) {
            // Extract the most relevant portion of the content
            $this->logSearch("processResults: Processing result #$index: " . $result['title']);
            $relevantContent = $this->extractRelevantContent($query, $result['content']);
            
            // Create an enhanced snippet
            $enhancedSnippet = $relevantContent ? $relevantContent : $result['snippet'];
            $this->logSearch("processResults: Enhanced snippet for #$index", [
                'original_length' => strlen($result['snippet']),
                'enhanced_length' => strlen($enhancedSnippet),
                'sample' => substr($enhancedSnippet, 0, 100) . '...'
            ]);
            
            $processedResults[] = [
                'title' => $result['title'],
                'link' => $result['link'],
                'snippet' => $enhancedSnippet,
                'sources' => isset($result['sources']) ? $result['sources'] : []
            ];
        }
        
        $this->logSearch("processResults: Complete", [
            'processed_count' => count($processedResults)
        ]);
        
        return $processedResults;
    }
    
    /**
     * Generate a comprehensive summary from multiple search results
     */
    private function generateSummary($query, $results) {
        $this->logSearch("generateSummary: Starting summary generation for query: $query");
        // Collect all content and track which sentences come from which sources
        $allSentences = [];
        $sentenceSources = [];
        $sources = [];
        foreach ($results as $index => $result) {
            if (!empty($result['content'])) {
                $sentences = preg_split('/(?<=[.!?])\s+/', $result['content']);
                foreach ($sentences as $sentence) {
                    $allSentences[] = $sentence;
                    $sentenceSources[] = [
                    'title' => $result['title'],
                    'url' => $result['link']
                ];
            }
        }
        }
        if (empty($allSentences)) {
            $this->logSearch("generateSummary: No content to summarize");
            return null;
        }
        // Score sentences for relevance to query
        $scores = [];
        $queryWords = preg_split('/\s+/', strtolower($query));
        foreach ($allSentences as $index => $sentence) {
            $scores[$index] = 0;
            $lowerSentence = strtolower($sentence);
            foreach ($queryWords as $word) {
                if (strlen($word) > 3 && strpos($lowerSentence, $word) !== false) {
                    $scores[$index] += 2;
                }
            }
            if (preg_match('/\b[A-Z][a-z]+ [A-Z][a-z]+\b/', $sentence)) {
                $scores[$index] += 1;
            }
            if (preg_match('/\b\d+\b/', $sentence)) {
                $scores[$index] += 1;
            }
            if (preg_match('/\b(January|February|March|April|May|June|July|August|September|October|November|December)\b/', $sentence)) {
                $scores[$index] += 1;
            }
            $wordCount = str_word_count($sentence);
            if ($wordCount < 5) {
                $scores[$index] -= 2;
            } elseif ($wordCount > 40) {
                $scores[$index] -= 1;
            }
        }
        // Get top sentences
        arsort($scores);
        $topIndices = array_keys(array_slice($scores, 0, 5, true));
        sort($topIndices);
        $summary = '';
        $usedSources = [];
        foreach ($topIndices as $index) {
            if (isset($allSentences[$index]) && strlen($allSentences[$index]) > 20) {
                $summary .= $allSentences[$index] . ' ';
                // Track the source for this sentence
                $src = $sentenceSources[$index];
                $usedSources[$src['url']] = $src; // deduplicate by URL
            }
        }
        $summary = trim($summary);
        // If summary is too short, add more sentences
        if (strlen($summary) < 100 && count($allSentences) > 5) {
            $additionalIndices = array_keys(array_slice($scores, 5, 5, true));
            sort($additionalIndices);
            foreach ($additionalIndices as $index) {
                if (isset($allSentences[$index]) && strlen($allSentences[$index]) > 20) {
                    $summary .= $allSentences[$index] . ' ';
                    $src = $sentenceSources[$index];
                    $usedSources[$src['url']] = $src;
                }
                if (strlen($summary) > 200) {
                    break;
                }
            }
        }
        $this->logSearch("generateSummary: Generated summary", [
            'summary_length' => strlen($summary),
            'sources_count' => count($usedSources)
        ]);
        // Post-process summary to add URLs to source names
        foreach ($usedSources as $src) {
            if (!empty($src['title']) && !empty($src['url'])) {
                $summary = preg_replace(
                    '/\\b' . preg_quote($src['title'], '/') . '\\b/u',
                    $src['title'] . ' (' . $src['url'] . ')',
                    $summary
                );
            }
        }
        return [
            'content' => $summary,
            'sources' => array_values($usedSources)
        ];
    }
    
    /**
     * Extract presidential information from search results
     */
    private function extractPresidentialInfo($results) {
        $presidentialInfo = null;
        $presidentNamePattern = '/\b(Joe Biden|Kamala Harris|Donald Trump|Barack Obama|George Bush|Bill Clinton)\b/i';
        
        foreach ($results as $result) {
            // Check if the content mentions a president
            if (preg_match($presidentNamePattern, $result['content'], $matches)) {
                $presidentName = $matches[1];
                
                // Find sentences mentioning the president
                $sentences = preg_split('/(?<=[.!?])\s+/', $result['content']);
                $relevantSentences = [];
                
                foreach ($sentences as $sentence) {
                    if (stripos($sentence, $presidentName) !== false) {
                        $relevantSentences[] = $sentence;
                    }
                }
                
                if (!empty($relevantSentences)) {
                    $presidentialInfo = [
                        'content' => implode(' ', $relevantSentences),
                        'president_name' => $presidentName,
                        'source_title' => $result['title'],
                        'source_url' => $result['link'],
                        'sources' => [
                            [
                                'title' => 'White House',
                                'url' => 'https://whitehouse.gov'
                            ]
                        ]
                    ];
                    
                    return $presidentialInfo;
                }
            }
        }
        
        return null;
    }
    
    /**
     * Extract the most relevant content from scraped text based on the query
     */
    private function extractRelevantContent($query, $content) {
        // Split content into sentences
        $sentences = preg_split('/(?<=[.!?])\s+/', $content);
        
        // Simple relevance scoring based on keyword matching
        $scores = [];
        $queryWords = preg_split('/\s+/', strtolower($query));
        
        foreach ($sentences as $index => $sentence) {
            $scores[$index] = 0;
            $lowerSentence = strtolower($sentence);
            
            foreach ($queryWords as $word) {
                if (strlen($word) > 3 && strpos($lowerSentence, $word) !== false) {
                    $scores[$index] += 1;
                }
            }
        }
        
        // Get the top 3 most relevant sentences
        arsort($scores);
        $topIndices = array_keys(array_slice($scores, 0, 3, true));
        sort($topIndices); // Re-sort by position in the original text
        
        // Combine the sentences
        $relevantContent = '';
        foreach ($topIndices as $index) {
            if (isset($sentences[$index])) {
                $relevantContent .= $sentences[$index] . ' ';
            }
        }
        
        return trim($relevantContent);
    }
    
    /**
     * Generate a cache key from a string
     */
    private function generateCacheKey($str) {
        return md5($str);
    }
    
    /**
     * Get clean text from the whole document
     */
    private function getCleanText(DomCrawler $crawler) {
        // Clone the crawler to avoid modifying the original
        $clone = clone $crawler;
        
        // Remove scripts, styles and comments
        $clone->filter('script, style, comment')->each(function (DomCrawler $node) {
            $node->getNode(0)->parentNode->removeChild($node->getNode(0));
        });
        
        // Get body text
        $text = $clone->filter('body')->text('');
        
        // Clean up whitespace
        $text = preg_replace('/\s+/', ' ', $text);
        
        return trim($text);
    }
    
    /**
     * Summarize text to a certain length
     */
    private function summarizeText($text, $length = 200) {
        if (strlen($text) <= $length) {
            return $text;
        }
        
        $text = substr($text, 0, $length);
        $pos = strrpos($text, ' ');
        
        if ($pos !== false) {
            $text = substr($text, 0, $pos);
        }
        
        return $text . '...';
    }
    
    /**
     * Convert relative URL to absolute URL
     */
    private function makeAbsoluteUrl($baseUrl, $relativeUrl) {
        // Parse base URL
        $parsedUrl = parse_url($baseUrl);
        $scheme = isset($parsedUrl['scheme']) ? $parsedUrl['scheme'] . '://' : 'https://';
        $host = isset($parsedUrl['host']) ? $parsedUrl['host'] : '';
        $path = isset($parsedUrl['path']) ? $parsedUrl['path'] : '';
        
        // Handle different relative URL formats
        if (strpos($relativeUrl, '//') === 0) {
            // Protocol-relative URL
            return $scheme . ltrim($relativeUrl, '/');
        } elseif (strpos($relativeUrl, '/') === 0) {
            // Root-relative URL
            return $scheme . $host . $relativeUrl;
        } else {
            // Directory-relative URL
            $dirPath = dirname($path);
            if ($dirPath !== '/') {
                $dirPath .= '/';
            }
            return $scheme . $host . $dirPath . $relativeUrl;
        }
    }
    
    /**
     * Flatten a scraped result tree into an array of attachments (screenshot+text)
     * @param array $result
     * @return array
     */
    public static function flattenScrapedResults($result) {
        $attachments = [];
        if (!empty($result['screenshot_base64']) && !empty($result['content'])) {
            $attachments[] = [
                'base64' => $result['screenshot_base64'],
                'text' => $result['content'],
                'url' => $result['url'],
                'title' => $result['title']
            ];
        }
        if (!empty($result['children']) && is_array($result['children'])) {
            foreach ($result['children'] as $child) {
                $attachments = array_merge($attachments, self::flattenScrapedResults($child));
            }
        }
        return $attachments;
    }
}
?> 